from fastapi import APIRouter, Depends, HTTPException, status
from sqlalchemy.orm import Session
from sqlalchemy import text
from typing import List
from uuid import uuid4

from db.db_connection import get_db
from db.models.order_model import Order, OrderCreate, OrderResponse
from db.models.lorry_model import Lorry

router = APIRouter(
    prefix="/orders",
    tags=["Orders"]
)

def get_order_or_404(db: Session, order_id: int):
    order = db.query(Order).filter(Order.id == order_id).first()
    if not order:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Order not found")
    return order


# ===============================
# Create Order
# ===============================
@router.post("/", response_model=OrderResponse, status_code=status.HTTP_201_CREATED)
def create_booking(
    order_data: OrderCreate,
    db: Session = Depends(get_db)
):
    if not order_data.email:
        raise HTTPException(status_code=400, detail="Email is required")

    booking_id = order_data.booking_id or uuid4().hex[:8]

    while db.query(Order).filter(Order.booking_id == booking_id).first():
        booking_id = str(uuid4())

    new_order = Order(
        user_id=order_data.user_id,
        booking_id=booking_id,
        origin=order_data.origin,
        destination=order_data.destination,
        move_date=order_data.move_date,
        distance_km=order_data.distance_km,
        lorry_type=order_data.lorry_type,  
        estimated_cost=order_data.estimated_cost,
        manpower=order_data.manpower,
        full_name=order_data.full_name,
        email=order_data.email,
        phone=order_data.phone,
        notes=order_data.notes,
    )

    try:
        db.add(new_order)
        db.commit()
        db.refresh(new_order)
        return new_order
    except Exception as e:
        db.rollback()
        raise HTTPException(status_code=500, detail=f"Internal error: {str(e)}")


# ===============================
# Get All Orders
# ===============================
@router.get("/", response_model=List[OrderResponse])
def get_orders(db: Session = Depends(get_db)):
    # Get orders with lorry images by joining with lorry table
    orders = db.query(Order).order_by(Order.created_at.desc()).all()
    
    # Add lorry images to each order
    for order in orders:
        lorry = db.query(Lorry).filter(Lorry.lorry_size == order.lorry_type).first()
        if lorry and lorry.images:
            # Add lorry_images field to the order object
            order.lorry_images = lorry.images
        else:
            order.lorry_images = []
    
    return orders


# ===============================
# Get Order by ID
# ===============================
@router.get("/{order_id}", response_model=OrderResponse)
def get_order_by_id(order_id: int, db: Session = Depends(get_db)):
    return get_order_or_404(db, order_id)


# ===============================
# Delete Order
# ===============================
@router.delete("/{order_id}", status_code=status.HTTP_204_NO_CONTENT)
def delete_order(order_id: int, db: Session = Depends(get_db)):
    order = get_order_or_404(db, order_id)
    db.delete(order)
    db.commit()
    return {"message": "Order deleted successfully"}
