from fastapi import FastAPI
from fastapi.middleware.cors import CORSMiddleware
from db.db_connection import init_table

# Import all SQLAlchemy models
from db.models.lorry_model import Lorry
from db.models.order_model import Order
from db.models.user_model import User

from routers import lorry_router, house_router 
from routers.AI import lorry_estimator_router, property_chatbot_router
from routers import location_search_router
from routers import auth_router 
from routers import order_router

init_table()

app = FastAPI(
    title="KitaMove API",
    description="API for managing lorries and houses",
    version="1.0.0"
)

# -----------------------------
# CORS middleware - Updated for better frontend integration
# -----------------------------
app.add_middleware(
    CORSMiddleware,
    allow_origins=[
        "http://localhost:3000",       # CRA or Vite dev server
        "http://localhost:5173",       # Vite default port
        "http://localhost:5175",       # Alternative frontend port
        "http://localhost:5176",       # Alternative frontend port
        "http://127.0.0.1:3000",       
        "http://127.0.0.1:5173",
        "http://127.0.0.1:5175",
        "http://127.0.0.1:5176",
        "http://localhost:5000",       # Proxy server
        "http://127.0.0.1:5000",
    ],
    allow_credentials=True,
    allow_methods=["GET", "POST", "PUT", "DELETE", "OPTIONS"], 
    allow_headers=["*"],
)

# -----------------------------
# Include Routers
# -----------------------------
app.include_router(lorry_router.router)
app.include_router(house_router.router)
app.include_router(lorry_estimator_router.router)
app.include_router(property_chatbot_router.router)
app.include_router(location_search_router.router) 
app.include_router(auth_router.router) 
app.include_router(order_router.router)

# -----------------------------
# Root endpoint
# -----------------------------
@app.get("/")
def read_root():
    return {"message": "Welcome to KitaMove API - Managing Lorries and Houses"}

# -----------------------------
# Health check endpoint
# -----------------------------
@app.get("/health")
def health_check():
    return {
        "status": "healthy",
        "service": "KitaMove API",
        "version": "1.0.0",
        "database": "connected"
    }

# -----------------------------
# API info endpoint
# -----------------------------
@app.get("/api/info")
def api_info():
    return {
        "name": "KitaMove API",
        "version": "1.0.0",
        "description": "Backend API for KitaMove moving services",
        "endpoints": {
            "houses": "/houses",
            "lorries": "/lorries", 
            "ai": "/ai",
            "auth": "/auth",      
            "health": "/health"
        }
    }
