import os
import sys
from sqlalchemy import create_engine, text
from dotenv import load_dotenv

# Add parent directory to path to import models
sys.path.append(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from db_connection import get_database_url

def populate_lorry_data():
    """Populate lorry data in PostgreSQL"""
    
    # Load environment variables
    load_dotenv()
    
    try:
        # Get database URL and create engine
        database_url = get_database_url()
        engine = create_engine(database_url)
        
        print("🔗 Connecting to database...")
        
        # Check if lorry table exists
        with engine.connect() as connection:
            result = connection.execute(text("""
                SELECT EXISTS (
                    SELECT FROM information_schema.tables 
                    WHERE table_schema = 'public' 
                    AND table_name = 'lorries'
                );
            """))
            
            table_exists = result.fetchone()[0]
            
            if not table_exists:
                print("❌ Lorries table does not exist!")
                print("Please run the database migrations first.")
                return False
        
        # Sample lorry data
        lorry_data = [
            {
                'name': 'Small Van',
                'capacity': '500kg',
                'dimensions': '2.5m x 1.5m x 1.5m',
                'price_per_km': 2.50,
                'description': 'Perfect for small moves and deliveries',
                'image_url': 'https://images.unsplash.com/photo-1560518883-ce09059eeffa?ixlib=rb-4.0.3&auto=format&fit=crop&w=500&q=80'
            },
            {
                'name': 'Medium Truck',
                'capacity': '2 tons',
                'dimensions': '4m x 2m x 2m',
                'price_per_km': 4.00,
                'description': 'Ideal for apartment moves and medium loads',
                'image_url': 'https://images.unsplash.com/photo-1560518883-ce09059eeffa?ixlib=rb-4.0.3&auto=format&fit=crop&w=500&q=80'
            },
            {
                'name': 'Large Truck',
                'capacity': '5 tons',
                'dimensions': '6m x 2.5m x 2.5m',
                'price_per_km': 6.50,
                'description': 'For large moves and commercial transport',
                'image_url': 'https://images.unsplash.com/photo-1560518883-ce09059eeffa?ixlib=rb-4.0.3&auto=format&fit=crop&w=500&q=80'
            },
            {
                'name': 'Furniture Van',
                'capacity': '3 tons',
                'dimensions': '5m x 2.2m x 2.2m',
                'price_per_km': 5.00,
                'description': 'Specialized for furniture and fragile items',
                'image_url': 'https://images.unsplash.com/photo-1560518883-ce09059eeffa?ixlib=rb-4.0.3&auto=format&fit=crop&w=500&q=80'
            },
            {
                'name': 'Pickup Truck',
                'capacity': '1 ton',
                'dimensions': '3m x 1.8m x 1.2m',
                'price_per_km': 3.00,
                'description': 'Versatile for small moves and deliveries',
                'image_url': 'https://images.unsplash.com/photo-1560518883-ce09059eeffa?ixlib=rb-4.0.3&auto=format&fit=crop&w=500&q=80'
            }
        ]
        
        print(f"📊 Found {len(lorry_data)} lorry types to populate")
        
        # Populate lorry data
        populated_count = 0
        with engine.connect() as connection:
            for lorry in lorry_data:
                try:
                    # Check if lorry already exists
                    result = connection.execute(text("""
                        SELECT id FROM lorries WHERE name = :name
                    """), {'name': lorry['name']})
                    
                    if result.fetchone():
                        print(f"⏭️  Skipping duplicate: {lorry['name']}")
                        continue
                    
                    # Insert new lorry
                    connection.execute(text("""
                        INSERT INTO lorries (
                            name, capacity, dimensions, price_per_km, 
                            description, image_url, is_available
                        ) VALUES (
                            :name, :capacity, :dimensions, :price_per_km,
                            :description, :image_url, :is_available
                        )
                    """), {
                        'name': lorry['name'],
                        'capacity': lorry['capacity'],
                        'dimensions': lorry['dimensions'],
                        'price_per_km': lorry['price_per_km'],
                        'description': lorry['description'],
                        'image_url': lorry['image_url'],
                        'is_available': True
                    })
                    
                    populated_count += 1
                    print(f"✅ Added: {lorry['name']}")
                        
                except Exception as e:
                    print(f"⚠️  Error adding lorry {lorry['name']}: {e}")
                    continue
            
            # Commit all changes
            connection.commit()
        
        print(f"✅ Successfully populated {populated_count} lorry types!")
        return True
        
    except Exception as e:
        print(f"❌ Population failed: {e}")
        return False

if __name__ == "__main__":
    print("🚛 Populating Lorry Data in PostgreSQL...")
    print("=" * 50)
    
    success = populate_lorry_data()
    
    print("=" * 50)
    if success:
        print("🎉 Lorry data population completed successfully!")
    else:
        print("💥 Lorry data population failed!")
        sys.exit(1)
