from pydantic import BaseModel
from typing import Optional, List

from sqlalchemy import Column, Integer, String, DECIMAL
from sqlalchemy.dialects.postgresql import JSONB
from db.db_connection import Base

# ===============================
# SQLAlchemy ORM Model
# ===============================
class Lorry(Base):
    __tablename__ = "lorry"  # table name in db

    id = Column(Integer, primary_key=True, autoincrement=True)
    lorry_size = Column(String(50), nullable=False)  
    base_price = Column(DECIMAL(10,2), nullable=False) 
    price_per_km = Column(DECIMAL(10,2), nullable=False)  # additional km rate
    weight_kg = Column(DECIMAL(10,2), nullable=True)  # max cargo weight (nullable for Van)
    length_m = Column(DECIMAL(5,2), nullable=False)
    width_m = Column(DECIMAL(5,2), nullable=False)
    height_m = Column(DECIMAL(5,2), nullable=False)
    images = Column(JSONB, nullable=True)    # Store image paths as JSONB (e.g. ["lorry1.jpg", "lorry2.jpg"])
    additional_manpower_price = Column(DECIMAL(10,2), nullable=False, default=0.0)
    man_power = Column(Integer, nullable=True)  # number of manpower available 


# ===============================
# Pydantic Model for Input
# ===============================
class LorryModel(BaseModel):
    lorry_size: str
    base_price: float
    price_per_km: float
    weight_kg: Optional[float] = None  # nullable for Van
    length_m: float
    width_m: float
    height_m: float
    images: Optional[List[str]] = None  # list of image paths
    additional_manpower_price: float = 0.0
    man_power: Optional[int] = None  # number of manpower available


# ===============================
# Pydantic Model for Response
# ===============================
class LorryResponse(BaseModel):
    id: int
    lorry_size: str
    base_price: float
    price_per_km: float
    weight_kg: Optional[float] = None
    length_m: float
    width_m: float
    height_m: float
    images: Optional[List[str]] = None
    additional_manpower_price: float
    man_power: Optional[int] = None

    class Config:
        from_attributes = True  # allows ORM -> Pydantic conversion
